/*
 * Decompiled with CFR 0.152.
 */
package org.n52.math;

import java.util.GregorianCalendar;
import org.n52.math.AbstractAlgorithm;
import org.n52.math.Algorithm;
import org.n52.math.TerrainTriangulation;
import org.n52.math.Vector3D;

public class SunTriangulation {
    final double aSun = 1.4959789E12;
    final double eSun = 0.017;
    private double lat = Double.NaN;
    private double lon = Double.NaN;
    private final double utc;
    private GregorianCalendar calendar;
    private int julianday;
    private final double yearFraction;
    private TerrainTriangulation terrTriangulation;
    private double zenithAngle;
    private double azimuth;
    private double distance;
    private boolean zenithAngleCalculated;
    private boolean azimuthCalculated;
    private boolean distanceCalculated;
    private static final String[] paramNames = new String[]{"lat", "lon"};
    private static final String[] paramDescr = new String[]{"latitude in degrees at terrain location", "longitude in degrees at terrain location"};

    public SunTriangulation(TerrainTriangulation terrTri, int year, int month, int dayInMonth, double utc) {
        this.terrTriangulation = terrTri;
        this.julianday = this.getJulianDay(year, month, dayInMonth);
        this.utc = utc;
        this.yearFraction = this.getYearFraction(year, this.julianday, utc);
        this.clear();
    }

    private int getJulianDay(int yr, int month, int dayInMonth) {
        this.calendar = new GregorianCalendar(yr, month - 1, dayInMonth);
        return this.calendar.get(6);
    }

    private double getYearFraction(int year, int julianday, double utc) {
        int daysInYear = 365;
        if (this.calendar.isLeapYear(year)) {
            ++daysInYear;
        }
        return Math.PI * 2 * ((double)(julianday - 1) + (utc - 12.0) / 24.0) / (double)daysInYear;
    }

    private double getMinutesShiftFromEquationOfTime(double gamma) {
        double EQT1 = 229.18;
        double EQT2 = 7.5E-5;
        double EQT3 = 0.001868;
        double EQT4 = 0.032077;
        double EQT5 = 0.014615;
        double EQT6 = 0.040849;
        double EquTime = 229.18 * (7.5E-5 + 0.001868 * Math.cos(gamma) - 0.032077 * Math.sin(gamma));
        double EquT2 = 229.18 * (-0.014615 * Math.cos(2.0 * gamma) - 0.040849 * Math.sin(2.0 * gamma));
        return EquTime += EquT2;
    }

    private double getTrueSolarTime(double lam, double utc, double eqtime_shift) {
        double time_offset = eqtime_shift - 4.0 * lam;
        return utc * 60.0 + time_offset;
    }

    private double getHourAngle(double trueSolarTime) {
        return (trueSolarTime / 4.0 - 180.0) * Math.PI / 180.0;
    }

    private double getSunDeclination(double gamma) {
        double DECL1 = 0.006918;
        double DECL2 = 0.399912;
        double DECL3 = 0.070257;
        double DECL4 = 0.006758;
        double DECL5 = 9.07E-4;
        double DECL6 = 0.002697;
        double DECL7 = 0.00148;
        double decli = 0.006918 - 0.399912 * Math.cos(gamma) + 0.070257 * Math.sin(gamma) - 0.006758 * Math.cos(2.0 * gamma) + 9.07E-4 * Math.sin(2.0 * gamma) - 0.002697 * Math.cos(3.0 * gamma) + 0.00148 * Math.sin(3.0 * gamma);
        return decli;
    }

    private double getSunDistance(double julianday) {
        double a = 1.4959789E12;
        double e = 0.017;
        double theta = Math.PI * 2 * (julianday - 1.0) / 365.25;
        double r = a * (1.0 - e * e) / (1.0 + e * Math.cos(theta));
        double E0 = 1.00011 + 0.034221 * Math.cos(theta) + 0.00128 * Math.sin(theta) + 7.19E-4 * Math.cos(2.0 * theta) + 7.7E-5 * Math.sin(2.0 * theta);
        return 1.0 / E0;
    }

    private Vector3D getSunPositionVector() {
        double eqt = this.getMinutesShiftFromEquationOfTime(this.yearFraction);
        double lamLocal = this.terrTriangulation.getLam();
        double trueSolarT = this.getTrueSolarTime(lamLocal, this.utc, eqt);
        double sunLam = -this.getHourAngle(trueSolarT);
        double sunPsi = this.getSunDeclination(this.yearFraction);
        Vector3D vec = new Vector3D();
        double posRadius = this.getSunDistance(this.yearFraction / 24.0);
        vec.set(0, (posRadius *= 1.4959789E12) * Math.cos(sunLam) * Math.cos(sunPsi));
        vec.set(1, posRadius * Math.sin(sunLam) * Math.cos(sunPsi));
        vec.set(2, posRadius * Math.sin(sunPsi));
        return vec;
    }

    private Vector3D getTerrainToSunVector(Vector3D vecTerrainPosition, Vector3D vecSunPosition) {
        return vecSunPosition.minus(vecTerrainPosition);
    }

    private void check(double lat, double lon) {
        if (this.lat != lat || this.lon != lon) {
            this.clear();
            this.lat = lat;
            this.lon = lon;
        }
    }

    private void clear() {
        this.zenithAngleCalculated = false;
        this.azimuthCalculated = false;
        this.distanceCalculated = false;
    }

    private double getZenithAngle(double lat, double lon) {
        this.check(lat, lon);
        if (!this.zenithAngleCalculated) {
            Vector3D unitV1 = this.terrTriangulation.getUnitVerticalVector(lat, lon);
            Vector3D posVecTerr = this.terrTriangulation.getTerrainPositionVector(lat, lon);
            Vector3D posVecSun = this.getSunPositionVector();
            Vector3D V2 = this.getTerrainToSunVector(posVecTerr, posVecSun);
            Vector3D unitV2 = V2.times(1.0 / V2.normF());
            this.zenithAngleCalculated = true;
            this.zenithAngle = Math.acos(unitV1.dotProduct(unitV2)) * 180.0 / Math.PI;
            if (Math.abs(this.zenithAngle) >= 90.0) {
                this.zenithAngle = Double.NaN;
            }
        }
        return this.zenithAngle;
    }

    private double getAzimuth(double lat, double lon) {
        this.check(lat, lon);
        if (!this.azimuthCalculated) {
            Vector3D unitNorth = this.terrTriangulation.getUnitNorthVector(lat, lon);
            Vector3D unitEast = this.terrTriangulation.getUnitEastVector(lat, lon);
            Vector3D posVecTerr = this.terrTriangulation.getTerrainPositionVector(lat, lon);
            Vector3D posVecSat = this.getSunPositionVector();
            Vector3D V2 = this.getTerrainToSunVector(posVecTerr, posVecSat);
            Vector3D unitV2 = V2.times(1.0 / V2.normF());
            double nort = unitNorth.dotProduct(unitV2);
            double east = unitEast.dotProduct(unitV2);
            this.azimuth = Math.atan2(east, nort) * 180.0 / Math.PI;
            if (this.azimuth < 0.0) {
                this.azimuth += 360.0;
            }
            this.azimuthCalculated = true;
        }
        return this.azimuth;
    }

    private double getDistance(double lat, double lon) {
        this.check(lat, lon);
        if (!this.distanceCalculated) {
            Vector3D posVecTerr = this.terrTriangulation.getTerrainPositionVector(lat, lon);
            Vector3D posVecSat = this.getSunPositionVector();
            Vector3D V2 = this.getTerrainToSunVector(posVecTerr, posVecSat);
            this.distanceCalculated = true;
            this.distance = V2.length();
            this.distance = this.getSunDistance(this.julianday);
        }
        return this.distance;
    }

    private double getEquationOfTimeResult(double lat, double lon) {
        return this.getMinutesShiftFromEquationOfTime(this.yearFraction);
    }

    private double getSunDeclinDegrees(double lat, double lon) {
        return this.getSunDeclination(this.yearFraction) * 180.0 / Math.PI;
    }

    public Algorithm getZenithAngleAlgorithm() {
        return new AbstractAlgorithm("Zenith Angle", "angle between local vertical and sun direction", paramNames, paramDescr){

            public double calculate(double[] params) {
                return SunTriangulation.this.getZenithAngle(params[0], params[1]);
            }
        };
    }

    public Algorithm getAzimuthAlgorithm() {
        return new AbstractAlgorithm("Azimuth", "angle betw. local North and projected sun direction", paramNames, paramDescr){

            public double calculate(double[] params) {
                return SunTriangulation.this.getAzimuth(params[0], params[1]);
            }
        };
    }

    public Algorithm getDistanceAlgorithm() {
        return new AbstractAlgorithm("Distance", "distance from terrain point to sun (in AU)", paramNames, paramDescr){

            public double calculate(double[] params) {
                return SunTriangulation.this.getDistance(params[0], params[1]);
            }
        };
    }

    public Algorithm getEquationOfTimeResultAlgorithm() {
        return new AbstractAlgorithm("EqOfTime", "deviation in minutes from regular solar position", paramNames, paramDescr){

            public double calculate(double[] params) {
                return SunTriangulation.this.getEquationOfTimeResult(params[0], params[1]);
            }
        };
    }

    public Algorithm getSunDeclinationAlgorithm() {
        return new AbstractAlgorithm("Declination", "declination of sun with r. to earth equator plane", paramNames, paramDescr){

            public double calculate(double[] params) {
                return SunTriangulation.this.getSunDeclinDegrees(params[0], params[1]);
            }
        };
    }
}

