import java.awt.Dimension;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;


public class GeneralConfig {
	
	private Value title;
	private Value sourceFolder;
	private Value otherFolder;
	private Value copyFiles;
	private Value autostartDelay;
	private int itemCols;
	
	private List<Sensor> sensors;
	private FormUtils fu;
	private ConfigFile configFile;
	
	private ActivityMonitor monitor;
	private LogWindow logWindow;
	
	private int dayNr;
	
	private static final String TITLE = "Title:";
	private static final String SOURCE_FOLDER = "Source folder:";
	private static final String OTHER_FILES_FOLDER = "Unmatched files folder:";
	private static final String COPY_FILES = "Copy files:";
	private static final String AUTOSTART_DELAY = "Autostart delay:";
	private static final String NR_COLUMNS = "Columns:";

	public GeneralConfig(ConfigFile configFile) {
		Calendar calendar = new GregorianCalendar();
		this.dayNr = calendar.get(Calendar.DAY_OF_MONTH); // avoid deleting on startup
		this.configFile = configFile;
		
		title = new Value("");
		sourceFolder = new Value("");
		otherFolder = new Value("");
		copyFiles = new Value(false);
		autostartDelay = new Value(10);
		
		sensors = new ArrayList<Sensor>();
		fu = new FormUtils(this);
		
		if (configFile.nrLines() > 0)
			read();
		else {
			configFile.newFile();
			configFile.writeToken("Group Name:", "Sample Group", null);
			configFile.writeToken("Item Name:", "Sample Item", null);
			configFile.rewind();
			read();
			store();
		}
	}

	public void read() {
		this.title.setVal(configFile.readToken(TITLE, "Data Manager"));
		this.sourceFolder.setVal(configFile.readToken(SOURCE_FOLDER, "C:\\source_folder"));
		this.otherFolder.setVal(configFile.readToken(OTHER_FILES_FOLDER, ""));
		this.copyFiles.setVal(configFile.readToken(COPY_FILES, "no").compareToIgnoreCase("yes") == 0);
		this.autostartDelay.setVal(configFile.readToken(AUTOSTART_DELAY, "10"));
		this.itemCols = Integer.parseInt(configFile.readToken(NR_COLUMNS, "2"));

		Sensor sensor = new Sensor();
		while (sensor.read(configFile)) {
			sensors.add(sensor);
			sensor = new Sensor();
		}		
	}
	
	public void write() {
		fu.updateData(true);
		configFile.writeToken("This file will be automatically overwritten the next time you run the program!", "", "");
		configFile.writeToken("You can make edits to this file when the program is not running", "", "");
		configFile.writeToken("but it is no use to change the layout of this file or add your own comments.", "", "");
		configFile.writeToken("", "", null);
		configFile.writeToken("Uncomment lines to let them take effect", "", "");
		configFile.writeToken("", "", null);
		configFile.writeToken(TITLE, title.getStringVal(), "Data Manager");
		configFile.writeToken(SOURCE_FOLDER, sourceFolder.getStringVal(), "C:\\source_folder");
		configFile.writeToken(OTHER_FILES_FOLDER, otherFolder.getStringVal(), "");
		configFile.writeToken(COPY_FILES, copyFiles.getBoolVal() ? "yes" : "no", "no");
		configFile.writeToken(AUTOSTART_DELAY, autostartDelay.getStringVal(), "10");
		configFile.writeToken(NR_COLUMNS, "" + itemCols, "2");
		configFile.writeln();
		for (Sensor sensor : sensors)
			sensor.write(configFile);
	}
	
	public String getTitle() {
		return title.getStringVal();
	}
	
	public int getAutostartDelay() {
		return autostartDelay.getIntVal();
	}

	public final JButton addToPanel(final Frame parent) {
		JPanel formPanel = fu.createGridPanel(null);
		
		// general input
		JPanel generalInput = fu.createGridPanel("Input folders");
		int ypos = 0;
		fu.addBrowseFolder(generalInput, SOURCE_FOLDER, sourceFolder, 0, ypos++);
		fu.addBrowseFolder(generalInput, OTHER_FILES_FOLDER, otherFolder, 0, ypos++);
		fu.addPanel(formPanel, generalInput, 0, 0);
//		formPanel.add(generalInput);

		// control
		ypos = 0;
		JPanel controlPanel = fu.createGridPanel("Activity");
		final JButton actionButton = fu.addActionButton(controlPanel, "Start", 1, ypos);
		final JButton showLogWindowButton = fu.addActionButton(controlPanel, "View Log", 2, ypos);
		JTextField activityText = fu.addTextBox(controlPanel, "", "", new Value(""), false, 0, ypos++);
		activityText.setMinimumSize(new Dimension(200,20));
		activityText.setPreferredSize(new Dimension(200,20));
//		JTextField messageText = fu.addTextBox(controlPanel, "", "", new Value(""), false, 2, ypos++);
		JProgressBar progressBar = fu.addProgressBar(controlPanel, 0, ypos);
		progressBar.setMinimumSize(new Dimension(200,20));
		progressBar.setPreferredSize(new Dimension(200,20));
		fu.addPanel(formPanel, controlPanel, 1, 0);
//		formPanel.add(controlPanel);
		monitor = new ActivityMonitor(progressBar, activityText);
		logWindow = new LogWindow(parent);

		// tabs
		JTabbedPane tabs = new JTabbedPane();
		for (Sensor sensor : sensors) {
			JPanel sensorPanel = new JPanel();
			sensor.addToPanel(sensorPanel, fu, itemCols);
			tabs.addTab(sensor.getName(), sensorPanel);
		}
		fu.addPanelEntireRow(formPanel, tabs, 0, 1);
//		formPanel.add(tabs);
		
		parent.add(formPanel);
		
		final GeneralConfig generalConfig = this;
		actionButton.addActionListener(new ActionListener() {
			FileProcessor processor = null;
			public void actionPerformed(ActionEvent event) {
				if (event.getSource() instanceof JButton) {
					if (processor == null) {
						fu.enableInput(false);
						fu.updateData(true);
						String msg = checkFolders(parent);
						if (msg.length() > 0) {
							messageBox(parent, "Incorrect folder specification", msg);
							fu.enableInput(true);
						} else {
							msg = checkData();
							if (msg.length() > 0) {
								messageBox(parent, "Incorrect input", msg);
								fu.enableInput(true);
							} else {
								actionButton.setText("Abort");
								logWindow.log("Started", false);
								processor = new FileProcessor(generalConfig, logWindow);
								processor.addActionListener(this);
							}
						}
					} else {
						processor.abort();
						actionButton.setText("Aborting...");
						actionButton.setEnabled(false);
					}
				} else if (event.getSource() instanceof FileProcessor) {
					String msg = event.getActionCommand();
					if (msg != null && msg.length() > 0 && msg.compareTo("Ready") != 0) {
						logWindow.log("Error: " + msg, true);
						messageBox(parent, "Error", msg);
					}
					processor = null;
					logWindow.log("Stopped", false);
					actionButton.setText("Start");
					actionButton.setEnabled(true);
					fu.enableInput(true);
				}
			}
		});
		showLogWindowButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				logWindow.ensureLogWindowShowing();
			}
		});
		parent.addComponentListener(new ComponentListener() {
			public void componentHidden(ComponentEvent event) {
			}

			public void componentMoved(ComponentEvent event) {
				logWindow.parentMoved();
			}

			public void componentResized(ComponentEvent event) {
			}

			public void componentShown(ComponentEvent event) {
			}
		});
		return actionButton;
	}
	
	public void processFiles() {
		// Init
		for (Sensor sensor : sensors)
			sensor.initFileProcessing();
		
		// Filter
		monitor.startJob("Reading source folder");
		File source = new File(sourceFolder.getStringVal());
		FilenameFilter filter = new FilenameFilter() {
			public boolean accept(File parent, String fileName) {
				File file = new File(parent, fileName);
				if (file.isDirectory())
					return false;

				for (Sensor sensor : sensors)
					if (sensor.acceptFile(file))
						return false;
				return true;
			}
			
		};
		File [] unprocessedFiles = source.listFiles(filter);
		monitor.endJob();
		
		boolean copy = copyFiles.getBoolVal();
		
		// Process
		for (Sensor sensor : sensors)
			sensor.processFiles(copy, monitor, logWindow);
		
		// Move "unmatched" files
		if (unprocessedFiles != null) {
			monitor.startJob(copy ? "Copying unmatched files" : "Moving unmatched files", unprocessedFiles.length);
			if (otherFolder.getStringVal() != null && otherFolder.getStringVal().length() > 0) {
				for (File file : unprocessedFiles) {
					File destinationFile = new File(new File(otherFolder.getStringVal()), file.getName());
					if (copy) {
						if (!destinationFile.exists()) {
							boolean success = FileProcessor.copyFile(file, destinationFile);
							if (!success) {
								logWindow.log("Could not copy " + file.getAbsolutePath() + " to " + otherFolder.getStringVal(), true);
								destinationFile.delete();
							}
						}
					} else {
						if (destinationFile.exists())
							destinationFile.delete();
						boolean success = file.renameTo(destinationFile);
						if (!success)
							logWindow.log((copy ? "Could not copy " : "Could not move ") + file.getAbsolutePath() + " to " + otherFolder.getStringVal(), true);
					}
					monitor.work();
				}
			}
			monitor.endJob();
		} else
			logWindow.log(copy ? "Could not copy unmatched files" : "Could not move unmatched files", true);

		// Delete old files at midnight
		Calendar calendar = new GregorianCalendar();
		int day = calendar.get(Calendar.DAY_OF_MONTH);
		if (day != dayNr) {
			dayNr = day;
			List<File> rootFolders = new ArrayList<File>();
			for (Sensor sensor : sensors)
				rootFolders.add(sensor.getRootFolder());
			for (Sensor sensor : sensors)
				sensor.deleteOldFiles(monitor, rootFolders, logWindow);
		}
	}

	public void store() {
		configFile.newFile();
		write();
		configFile.store();
	}
	
	private String checkFolders(Frame parent) {
		List<Tuple> nonExistingFolders = new ArrayList<Tuple>();
		File folder = new File(otherFolder.getStringVal());
		if (!folder.exists()) {
			Tuple tuple = new Tuple(folder, "Unmatched files");
			nonExistingFolders.add(tuple);
		}
		
		for (Sensor sensor : sensors) {	
			List<Tuple> tuples = sensor.getDestinationFoldersToBeCreated();
			if (tuples != null && tuples.size() > 0) {
				for (Tuple tuple : tuples)
					if (!nonExistingFolders.contains(tuple))
						nonExistingFolders.add(tuple);
			}
		}
		
		if (nonExistingFolders.size() > 0) {
			String message = "";
			for (Tuple tuple : nonExistingFolders) {
				if (tuple.getFolder().getPath().length() == 0) {
					if (message.length() > 0)
						message += ", ";
					message += tuple.getDataSetName();
				}
			}
			if (message.length() > 0) {
				message = "No folder specified for:\n" + message;
				return message;
			}
		}
		
		if (nonExistingFolders.size() > 0) {
			String message = "";
			for (Tuple tuple : nonExistingFolders)
				message += tuple.getDataSetName() + ": folder '" + tuple.getFolder().getAbsolutePath() + "' does not exist.\n" ;
			message += "Create all folders?";
			if (yesNoBox(parent, "Output folders do not exist", message)) {
				for (Tuple tuple : nonExistingFolders) {
					if (!tuple.getFolder().exists()) { // re-verify, as it might have been created by a previous tuple, and mkdirs returns false instead of unnecessary creating the folder
						if (!tuple.getFolder().mkdirs()) {
							messageBox(parent, "Failed to create folder", tuple.getDataSetName() + ": failed to create folder '" + tuple.getFolder().getAbsolutePath() + "'");
							return "Not all output folders have been created";						
						}
					}
				}
				messageBox(parent, "Output folders created.", "All output folders have been successfully created.");
			} else
				return "Output folders have not been created.";
		}
		
		return "";
	}
	
	private String checkData() {
		if (!new File(sourceFolder.getStringVal()).exists())
			return "Please ensure that the source folder exists.";

		if (!new File(otherFolder.getStringVal()).exists())
			return "Please ensure that the folder for unmatched files exists.";

		for (Sensor sensor : sensors) {
			String msg = sensor.checkData();
			if (msg.length() > 0)
				return msg;			
		}
		
		// All OK
		return "";
	}
	
	private void messageBox(Frame parent, String title, String msg) {
		JOptionPane.showMessageDialog(parent, msg, title, JOptionPane.WARNING_MESSAGE);
	}
	
	private boolean yesNoBox(Frame parent, String title, String msg) {
		return JOptionPane.YES_OPTION == JOptionPane.showConfirmDialog(parent, msg, title, JOptionPane.YES_NO_OPTION);
	}
}
