import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.border.Border;

public class FormUtils {
	
	private static final int BUTTON_XSIZE = 100;
	private static final int BROWSEFIELD_XSIZE = 300;
	private static final int TEXTFIELD_XSIZE = 150;
	private static final int COMPONENT_XSIZE = 150;
	private static final int COMPONENT_YSIZE = 20;
	
	public File lastFileChosen;
	public JFileChooser chooser;
	private GeneralConfig config;
	private GridBagConstraints layoutConstraints;
	private List<Component> inputComponents;
	private Map<Component,Value> valueComponents;
	
	public FormUtils(GeneralConfig config) {
		this.config = config;
		lastFileChosen = new File("C:\\");
		chooser = new JFileChooser();
		layoutConstraints = new GridBagConstraints();
		inputComponents = new ArrayList<Component>();
		valueComponents = new HashMap<Component,Value>();
	}
	
	public static String listToString(List<String> patternList) {
		String patterns = "";
		boolean first = true;
		if (patternList != null) {
			for (String pattern : patternList) {
				if (first)
					first = false;
				else
					patterns += ',';
				patterns += pattern;
			}
		}
		return patterns;
	}
	
	public static List<String> stringToList(String patterns) {
		List<String>patternList = new ArrayList<String>();
		if (patterns != null) {
			String [] tokens = patterns.split("[\\s*,\\s*]|[\\s+]");
			if (tokens != null) {
				for (String token : tokens)
					patternList.add(token);
			}			
		}
		return patternList;
	}

	public JTextField addTextBox(JPanel panel, String label, String tip, Value value, boolean editable, int xpos, int ypos) {
		if (label != null && label.length() > 0) {
			JLabel textLabel = new JLabel(label);
			textLabel.setToolTipText(tip);
			panel.add(textLabel, constraints(xpos++, ypos));
		}
		final JTextField textField = new JTextField(value.getStringVal())
		{
			@Override
			public String getToolTipText(MouseEvent event) {
				String tip = getText();
				if (tip != null && tip.length() > 0)
					return formatToolTip(getText());
				else
					return null;
			}
		};
    	textField.setToolTipText("dummy"); // otherwise it doesn't display tooltips
    	textField.setMinimumSize(new Dimension(TEXTFIELD_XSIZE,COMPONENT_YSIZE));
    	textField.setPreferredSize(new Dimension(TEXTFIELD_XSIZE,COMPONENT_YSIZE));
    	if (!editable) {
    		textField.setEditable(false);
    		textField.setForeground(Color.BLUE);
    		textField.setBackground(Color.LIGHT_GRAY);
    	} else {
    		inputComponents.add(textField);
    		valueComponents.put(textField, value);
    	}
    	panel.add(textField, constraints(xpos, ypos));
    	return textField;
	}
	
	public void addBrowseFolder(final JPanel panel, String label, Value value, int xpos, int ypos) {
		panel.add(new JLabel(label), constraints(xpos, ypos));
		final JTextField textField = new JTextField(value.getStringVal())
		{
			@Override
			public String getToolTipText(MouseEvent event) {
				String tip = getText();
				if (tip != null && tip.length() > 0)
					return formatToolTip(getText());
				else
					return null;
			}
		};
    	textField.setToolTipText("dummy"); // otherwise it doesn't display tooltips
    	textField.setMinimumSize(new Dimension(BROWSEFIELD_XSIZE,COMPONENT_YSIZE));
    	textField.setPreferredSize(new Dimension(BROWSEFIELD_XSIZE,COMPONENT_YSIZE));
    	inputComponents.add(textField);
    	valueComponents.put(textField, value);
    	panel.add(textField, constraints(xpos + 1, ypos));

		JButton browseButton = new JButton("Browse...");
		browseButton.setMinimumSize(new Dimension(BUTTON_XSIZE,COMPONENT_YSIZE));
		browseButton.setPreferredSize(new Dimension(BUTTON_XSIZE,COMPONENT_YSIZE));
		browseButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				String currentFile = textField.getText();
				if (currentFile.length() > 0) {
					File folder = getNearestFolder(currentFile);
					chooser.setCurrentDirectory(folder);
				}
				else
					chooser.setCurrentDirectory(lastFileChosen);
				chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int returnVal = chooser.showOpenDialog(panel);
			    if(returnVal == JFileChooser.APPROVE_OPTION) {
			          textField.setText(chooser.getSelectedFile().getAbsolutePath());
			          lastFileChosen = chooser.getSelectedFile();
			          storeConfig();
			    }
			}
		});
		inputComponents.add(browseButton);
		panel.add(browseButton, constraints(xpos + 2, ypos));
	}

	public void addBrowseFile(final JPanel panel, String label, Value value, int xpos, int ypos) {
		panel.add(new JLabel(label), constraints(xpos, ypos));
		final JTextField textField = new JTextField(value.getStringVal())
		{
			@Override
			public String getToolTipText(MouseEvent event) {
				String tip = getText();
				if (tip != null && tip.length() > 0)
					return formatToolTip(getText());
				else
					return null;
			}
		};
    	textField.setToolTipText("dummy"); // otherwise it doesn't display tooltips
    	textField.setMinimumSize(new Dimension(BROWSEFIELD_XSIZE,COMPONENT_YSIZE));
    	textField.setPreferredSize(new Dimension(BROWSEFIELD_XSIZE,COMPONENT_YSIZE));
    	inputComponents.add(textField);
    	valueComponents.put(textField, value);
    	panel.add(textField, constraints(xpos + 1, ypos));

		JButton browseButton = new JButton("Browse...");
		browseButton.setMinimumSize(new Dimension(BUTTON_XSIZE,COMPONENT_YSIZE));
		browseButton.setPreferredSize(new Dimension(BUTTON_XSIZE,COMPONENT_YSIZE));
		browseButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				String currentFile = textField.getText();
				if (currentFile.length() > 0) {
					File file = new File(currentFile);
					if (file.exists())
						chooser.setSelectedFile(file);
					else {
						file = getNearestFolder(currentFile);
						chooser.setCurrentDirectory(file);
					}
				}
				else
					chooser.setCurrentDirectory(lastFileChosen);
				chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
				int returnVal = chooser.showOpenDialog(panel);
			    if(returnVal == JFileChooser.APPROVE_OPTION) {
			          textField.setText(chooser.getSelectedFile().getAbsolutePath());
			          lastFileChosen = chooser.getSelectedFile();
			          storeConfig();
			    }
			}
		});
		inputComponents.add(browseButton);
		panel.add(browseButton, constraints(xpos + 2, ypos));

		JButton openButton = new JButton("Open");
		openButton.setMinimumSize(new Dimension(BUTTON_XSIZE,COMPONENT_YSIZE));
		openButton.setPreferredSize(new Dimension(BUTTON_XSIZE,COMPONENT_YSIZE));
		openButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				String currentFile = textField.getText();
				if ((currentFile.length() > 0) && (new File(currentFile).exists())) {
					try {
						Runtime.getRuntime().exec("notepad.exe " + currentFile);
					} catch (IOException e) {
					} finally {
						
					}
					
				}
			}
		});
		panel.add(openButton, constraints(xpos + 3, ypos));
	}
	
	private File getNearestFolder(String fileName) {
		File folder = new File(fileName);
		if (!folder.isDirectory())
			folder = folder.getParentFile();
		while (!folder.exists() && folder.getParentFile() != null)
			folder = folder.getParentFile();
		if (folder.exists())
			return folder;
		else
			return lastFileChosen;
	}

	private static String formatToolTip(String text) {
		final int maxCharsInLine = 80;
		String [] tokens = text.split("[\\s*,\\s*]|[\\s+]");
		if (tokens != null && tokens.length > 1) {
			String ret = "<html>";
			String line = "";
			boolean firstLineInText = true;
			boolean firstTokenInLine = true;
			for (int i = 0; i < tokens.length; ++i) {
				if (firstTokenInLine)
					firstTokenInLine = false;
				else
					line += ",";					
				line += tokens[i];
				if (line.length() > maxCharsInLine) {
					if (firstLineInText)
						firstLineInText = false;
					else
						ret += "<br>";
					ret += line;
					if (i < tokens.length)
						ret += ",";
					line = "";
					firstTokenInLine = true;
				}
			}
			// remainder
			if (line.length() > 0) {
				if (firstLineInText)
					firstLineInText = false;
				else
					ret += "<br>";
				ret += line;
				line = "";
				firstTokenInLine = true;
			}
			ret += "</html>";
			return ret;
		} else
			return text;
	}

	public JComboBox addComboBox(JPanel panel, String label, Value value, Object[] items, int xpos, int ypos) {
		panel.add(new JLabel(label), constraints(xpos, ypos));
		JComboBox combo = new JComboBox(items);
		combo.setSelectedItem(value.getObjectVal());
		combo.setMinimumSize(new Dimension(COMPONENT_XSIZE,COMPONENT_YSIZE));
		combo.setPreferredSize(new Dimension(COMPONENT_XSIZE,COMPONENT_YSIZE));
		combo.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				storeConfig();
			}
		});
		inputComponents.add(combo);
		valueComponents.put(combo, value);
		panel.add(combo, constraints(xpos + 1, ypos));
		return combo;
	}

	public void addCheckBox(JPanel panel, String label, Value value, int xpos, int ypos) {
		if (label.endsWith(":"))
			label = label.substring(0, label.length() - 1);
		JCheckBox checkBox = new JCheckBox(label, value.getBoolVal());
		checkBox.setMinimumSize(new Dimension(COMPONENT_XSIZE,COMPONENT_YSIZE));
		checkBox.setPreferredSize(new Dimension(COMPONENT_XSIZE,COMPONENT_YSIZE));
		checkBox.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				storeConfig();
			}
		});
		inputComponents.add(checkBox);
		valueComponents.put(checkBox, value);
		panel.add(checkBox, constraints(xpos, ypos));
	}
	
	public void addLabel(JPanel panel, String label, int xpos, int ypos) {
		panel.add(new JLabel(label), constraints(xpos, ypos));
	}
	
	public JButton addActionButton(JPanel panel, String label, int xpos, int ypos) {
		JButton actionButton = new JButton(label);
		actionButton.setMinimumSize(new Dimension(BUTTON_XSIZE,2*COMPONENT_YSIZE));
		actionButton.setPreferredSize(new Dimension(BUTTON_XSIZE,2*COMPONENT_YSIZE));
		actionButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				storeConfig();
			}
		});
		layoutConstraints = constraints(xpos, ypos);
//		layoutConstraints.gridwidth = 2;
		layoutConstraints.gridheight = 2;		
		panel.add(actionButton, layoutConstraints);
		return actionButton;
	}
	
	public JProgressBar addProgressBar(JPanel panel, int xpos, int ypos) {
		JProgressBar progressBar = new JProgressBar();
    	progressBar.setMinimumSize(new Dimension(TEXTFIELD_XSIZE,COMPONENT_YSIZE));
    	progressBar.setPreferredSize(new Dimension(TEXTFIELD_XSIZE,COMPONENT_YSIZE));
		layoutConstraints = constraints(xpos, ypos);
//		layoutConstraints.gridwidth = 3;
    	panel.add(progressBar, layoutConstraints);
    	return progressBar;
	}
	
	public void addPanel(Container parent, Component child, int xpos, int ypos) {
		parent.add(child, constraints(xpos, ypos));
	}
	
	public void addPanelEntireRow(Container parent, Component child, int xpos, int ypos) {
		layoutConstraints = constraints(xpos, ypos);
		layoutConstraints.gridwidth = GridBagConstraints.REMAINDER;
		parent.add(child, layoutConstraints);
	}

	private GridBagConstraints constraints(int xpos, int ypos) {
		layoutConstraints.gridx = xpos;
		layoutConstraints.gridy = ypos;
		layoutConstraints.gridwidth = 1;
		layoutConstraints.gridheight = 1;
		layoutConstraints.fill = GridBagConstraints.NONE;
		layoutConstraints.ipadx = 0;
		layoutConstraints.ipady = 0;
		layoutConstraints.insets = new Insets(1, 1, 1, 1);
		layoutConstraints.anchor = GridBagConstraints.WEST;
		layoutConstraints.weightx = 0.0;
		layoutConstraints.weighty = 0.0;
		
		return layoutConstraints;
	}

	public JPanel createGridPanel(String title) {
		Border border = BorderFactory.createRaisedBevelBorder();
		if (title != null && title.length() > 0)
			border = BorderFactory.createTitledBorder(border, title);
		JPanel panel = new JPanel();
		panel.setBorder (border);
		panel.setLayout(new GridBagLayout());
		return panel;
	}

	public JPanel createVerticalPanel(String title) {
		Border border = BorderFactory.createRaisedBevelBorder();
		if (title != null && title.length() > 0)
			border = BorderFactory.createTitledBorder(border, title);
		JPanel panel = new JPanel();
		panel.setBorder (border);
		panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
		return panel;
	}
	
	public void enableInput(boolean enabled) {
		for (Component component : inputComponents)
			component.setEnabled(enabled);				
	}

	/**
	 * 
	 * @param read indicates that values should be read from components
	 * read = true means component's states will be recorded into values
	 * read = false means values' states will be recorded into components
	 */
	public void updateData(boolean read) {
		for (Component component : valueComponents.keySet()) {
			Value value = valueComponents.get(component);
			if (read)
				value.setValFrom(component);
			else
				value.putValInto(component);			
		}
	}
	
	private void storeConfig() {
		updateData(true);
		config.store();
	}
}
